package fccsc.manager;

/**
 * PBSC History
 * ************
 * PBSC palermor IR84565 05/19/2015 Update to log4j2.
 * PBSC palermor IR84565 06/03/2015 Fix issue where the size value does not get updated when there is an error.
 *                                  This error was reported by FLVC when testing a new version of Message Director.
 */

import intarsys.util.PropertyManager;

import java.util.Hashtable;
import java.util.Properties;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import fccsc.manager.data.ReturnCodes;
import fccsc.manager.data.edi.ControlBlock;
import fccsc.manager.data.edi.StandardDataBlock;
import fccsc.manager.data.process.ProcessFEDI;
import fccsc.manager.data.process.ProcessFXML;
import fccsc.manager.data.process.ProcessUnknownFEDI;
import fccsc.manager.data.process.ProcessUnknownFXML;
import fccsc.manager.data.xml.FactsMessage;


public final class MessageHandler
{
    private static Logger logger = LogManager.getLogger( MessageHandler.class );  // PBSC palermor IR84565

	private static Hashtable<String, MessageProperties> hashMessageProperties = new Hashtable<String, MessageProperties>( 10, 2 );

	private String            data    = "";
	private ControlBlock      ediCB   = null;
	private StandardDataBlock ediSDB  = null;
	private IMessage          message = null;
	private FactsMessage      xmlFM   = null;

	private Class<?>             cls     = null;
	private String            process = "";
	private MessageProperties mprop   = null;



	/**
	 * Creates a MessageHandler object.
	 */
    public
	MessageHandler()
    {
    }


	/**
	 * Returns the IMessage object being processed.
	 * @return an object which implements IMessage or null.
	 */
	public IMessage getMessage() { return this.message; }


	/**
	 *
	 * @param p_data
	 */
	public void
	processRequest( String p_data )
	{
		this.data = p_data;

		try
		{
			this.ediCB = new ControlBlock( this.data );

			if ( ediCB.isFEDI() )
			{
				ediSDB  = new StandardDataBlock( this.data );
				process = ediSDB.getProcess().trim();

				// determine if incoming process is any of the following ...
				//     VERIFY, TRANSCRIPT, COURSELIST, GRADAUDIT
				//     LOCALSHOP, REMOTESHOP, FACTSSHOP
				// else process an "Unknown" EDI process
		    }
			else if ( ediCB.isFXML() )
			{
				xmlFM   = new FactsMessage( this.data );
				process = xmlFM.getProcess().trim();

				// determine if incoming process is any of the following ...
				//     ADMISSION, 22ADVISE
				// else process an "Unknown" XML process
			}
			else
			{
				logger.error( "Unknown data type. The following message is not FEDI nor FXML:\n[{}]", this.data );  // PBSC palermor IR84565
			}

			loadProperties( process );

			if ( hashMessageProperties.containsKey( process ) )
			{
				mprop        = hashMessageProperties.get( process );
				cls          = (Class<?>) Class.forName( mprop.getObject() );
				this.message = (IMessage) cls.newInstance();
			}
			else
			{
				if ( ediCB.isFEDI() ) { this.message = (IMessage) new ProcessUnknownFEDI(); }
				else                  { this.message = (IMessage) new ProcessUnknownFXML(); }
			}

			this.message.setRequest( this.data );
	    	this.message.process();
		}
		catch ( Exception ex )
		{
		    String msg  = "ERROR [" + ReturnCodes.CODE_00032 + "-" + ReturnCodes.CODE_00032_MESSAGE + "]";
			String msgX = ex.toString();

		    System.out.println( "Processing (" + process + ") " + msgX );

			logger.fatal( msg  );
			logger.fatal( msgX );


		    ////////////////////////////////////////////////////
			// since an exception occurred ...
			// probably due to not being able to connect to broker
			// we must take the original message and plug-in
			// return code and message
			//
		    StringBuffer response = new StringBuffer();

			if ( ediCB.isFEDI() )
			{
				((ProcessFEDI) message).getEDIStandardDataBlock().setReturnCode(    ReturnCodes.CODE_00032 );
				((ProcessFEDI) message).getEDIStandardDataBlock().setReturnMessage( ReturnCodes.CODE_00032_MESSAGE );

				String extra = "";
				try { extra = message.getRequest().substring( ControlBlock.SIZE + StandardDataBlock.SIZE ); }
			    catch (Exception ex2) { logger.error( ex2.getMessage() ); }
				
				response.append( ((ProcessFEDI) message).getEDIControlBlock().getData() +
							     ((ProcessFEDI) message).getEDIStandardDataBlock().getData() +
								 extra );
		    }
			else if ( ediCB.isFXML() )
			{
				// PBSC palermor IR84565 START
				ControlBlock controlBlock = ((ProcessFXML) message).getEDIControlBlock();
				FactsMessage factsMessage = ((ProcessFXML) message).getXMLFactsMessage();

				factsMessage.setRequest(       "N"      );  // PLOG 23579
				factsMessage.setReturnCode(    ReturnCodes.CODE_00032 );
				factsMessage.setReturnMessage( ReturnCodes.CODE_00032_MESSAGE );

				// Set the new length.
				int totalLength = ControlBlock.SIZE + factsMessage.getXML().length();
				
				controlBlock.setTotalRecordSize(String.valueOf(totalLength));

				response.append( controlBlock.getData() + factsMessage.getXML()  );
				// PBSC palermor IR84565 END
			}
			try { message.setResponse( response.toString() ); }
			catch (Exception ex2) { logger.error( ex2.getMessage() ); }
		}
	}


	private void
	loadProperties( String p_process )
	{
		p_process.trim().toUpperCase();

		if ( ! hashMessageProperties.containsKey( p_process ) )
		{
		    Properties prop = PropertyManager.getProperty(
								    "message." + p_process + ".properties" );
		    if ( prop != null )
		    {
				MessageProperties mprop = new MessageProperties( p_process, prop );

			    hashMessageProperties.put( p_process, mprop );
				//System.out.println( mprop.toString() );
				logger.info( "Loaded properties for [{}]", p_process );  // PBSC palermor IR84565
		    }
			else
			{
				logger.error( "Cannot load properties file for [{}]", p_process );  // PBSC palermor IR84565
			}
		}
	}
}