package fccsc.manager;

/**
 * PBSC History
 * ************
 * PBSC palermor 167972 10/02/2013 Change location of property files.
 * PBSC palermor IR84565 05/15/2015 Update to use Log4j2.
 */

import intarsys.util.PropertyManager;

import java.io.File;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.Properties;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import fccsc.manager.crypto.MessageCryptography;
import fccsc.manager.util.TemplateManager;


public final class MessageManager
	extends Thread
{
    private static Logger logger = LogManager.getLogger( MessageManager.class ); // PBSC palermor IR84565

	private static ThreadGroup threadGroup = new ThreadGroup( "BALONIE" );
	private static boolean     listening   = false;

	private ServerSocket server    = null;
	private Socket       client    = null;


	/**
	 * Entry point for application.
	 *
	 * @param args no parameters required.
	 */
	public static void
	main( String [] args )
	{
		new MessageManager();
	}


	/**
	 * Instantiates a new MessageManager object and performs the following:
	 * - loads configuration data from property files,
	 * - loads data wrapping templates for required message processes,
	 * - and creates / manages *.logs files for logging events.
	 *
	 * The following required directories must exist relative to the current directory:
	 *
	 * /properties - contains all configuration *.properties files
	 * /templates  - contains all data wrapping *.template files
	 * /logs       - location for all *.logs files to be managed by Log4j
	 */
    public
	MessageManager()
	{
		try
		{
			///////////////////////////////////////////////////////////////
			// get the current directory path
			File f = new File( "." );

			///////////////////////////////////////////////////////////////
			// load the templates into memory
			//
			TemplateManager.init( f.getCanonicalPath() + File.separator + "templates" );
			TemplateManager.loadAll();

			///////////////////////////////////////////////////////////////
			// load the properties files
			// and fetch parameter values for this object's construction
			//
			PropertyManager.init( f.getCanonicalPath() + File.separator + "bin" ); // PBSC palermor 167972 Change to bin directory from properties
			PropertyManager.loadAll();

			///////////////////////////////////////////////////////////////
			// load the log4j properties
			//
			// PBSC palermor IR84565 START
			//Properties propLog4j = PropertyManager.getProperty( "log4j.properties" );
		    //PropertyConfigurator.configure( propLog4j );
			// PBSC palermor IR84565 END

			///////////////////////////////////////////////////////////////
			// initialize our cryptography object
			//
			Properties propCrypto = PropertyManager.getProperty( "cryptography.properties" );
		    MessageCryptography.init( propCrypto );

			///////////////////////////////////////////////////////////////
			// load property object and read parameters
			//
			Properties prop    = PropertyManager.getProperty( "manager.properties" );
			int        port    = Integer.parseInt( prop.getProperty( "manager.listener.port"    ) );
			int        backLog = Integer.parseInt( prop.getProperty( "manager.listener.backlog" ) );

			server    = new ServerSocket( port, backLog );
			listening = true;

		    String address = InetAddress.getLocalHost().getHostAddress();
			String msg     =    "Listening on address [" + address + "] and " +
							    "port [" + port + "] with a backlog of [" + backLog +"].";

			System.out.println( msg );
			logger.info( " " );
		    logger.info( "********************************************************************************" );
			logger.info( msg );
		    logger.info( "********************************************************************************" );
			logger.info( " " );

		    //if ( logger.isDebugEnabled() )
			//{
				//logger.debug( msg );
				//logger.debug( TemplateManager.display() );
				//logger.debug( PropertyManager.display() );
			//}

			this.start();
		}
		catch (Exception e)
		{
			listening = false;
			e.printStackTrace();

			logger.fatal( e.getMessage() );

			doShutdown();
		}
	}


	/**
	 * This is where this server's thread does all of
	 * its listening and handling for incoming requests.
	 * This process automatically "starts" when this object is instantiated.
	 */
	public void
	run()
	{
		try
		{
			Thread        thread  = null;
			MessageThread message = null;

			while ( (listening) && (server != null) )
			{
				client  = server.accept();
				message = new MessageThread( client );
				thread  = new Thread( threadGroup, message );
				thread.start();

				if ( logger.isDebugEnabled() )
				{
					logger.debug( "Starting a new MessageThread." );
				}
			}
		}
		catch (Exception e)
		{
			e.printStackTrace();
			logger.fatal( e.getMessage() );
		}
		finally
		{
			try
			{
				if ( client != null ) { client.close(); }
				if ( server != null ) { server.close(); }
			}
			catch (Exception e2)
			{
				e2.printStackTrace();
				logger.fatal( e2.getMessage() );
			}
			finally
			{
				doShutdown();
			}
		}
	}


	/**
	 * Performs a shutdown of this server.
	 * First, it triggers for the server to stop listening for requests.
	 * Then, it waits until all current threads complete their tasks.
	 * Finally, it shuts the server off and exits the JVM.
	 */
	public static void
	doShutdown()
	{
		doShutdown( null );
	}


	/**
	 * Performs a shutdown of this server.
	 * First, it triggers for the server to stop listening for requests.
	 * Then, it waits until all current threads complete their tasks.
	 * Finally, it shuts the server off and exits the JVM.
	 *
	 * @param p_params optional parameters ... (currently ignored).
	 */
	public static void
	doShutdown( String [] p_params )
	{
		System.out.println( "Shutdown process initiated ..." );
		logger.info(        "Shutdown process initiated ..." );

		try
		{
			listening = false;

			while ( threadGroup.activeCount() > 0 )
			{
				//System.out.println( "sleeping ..." );
				Thread.sleep( 250 );
			}
		}
		catch (Exception e)
		{
			e.printStackTrace();
			logger.error( "Shutdown process caused an error: {}", e.getMessage() );  // PBSC palermor IR84565
		}

		System.out.println( "Shutdown process completed successfully." );
		logger.info(        "Shutdown process completed successfully." );
		System.exit( 0 );
	}
}