package fccsc.manager.broker;

/**
 * PBSC History
 * ************
 * PBSC palermor IR84565 05/19/2015 Update to log4j2.
 * PBSC palermor IR84565 06/02/2015 Add to finally block to endAllConversations, logoff then disconnect as per documentation.
 */

import java.util.Properties;
import java.util.Vector;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.softwareag.entirex.aci.Broker;
import com.softwareag.entirex.aci.BrokerException;
import com.softwareag.entirex.aci.BrokerMessage;
import com.softwareag.entirex.aci.BrokerService;
import com.softwareag.entirex.aci.Conversation;
import com.softwareag.entirex.aci.UnitofWork;


/**
 * A <i>EntireXBroker</i> object encapsulates the main communication
 * gateway to a mainframe via an EntireX Java interface client
 * that communicates to a broker server module running on a mainframe.
 *
 * <p>
 * The following is sample code on how-to create send and receive
 * a message through broker:
 * <p><hr><blockquote><pre>
 *
 *  // create a broker communication interface
 * EntireXBroker broker = new EntireXBroker(
 * 							"etb227:9999", "UTLI01P1", "DEVL",
 * 							"STSI01N0", "BLAH", false, "30S", 30000 );
 *
 * OR use a property file with required parameters ...
 *
 * Properties    prop   = load properties file ...
 * EntireXBroker broker = new EntireXBroker( prop );
 *
 * // send request message
 *  broker.sendMessage( "<xml message ... >" );
 *
 * // get response message
 * String response = broker.getResponse();
 *
 *
 * Broker exception information written to log files and standard out:
 * -------------------------------------------------------------------
 * Format:    Broker Error [CLASS] [CODE] [MESSAGE]
 * Examples:  Broker Error 0002 0002: User does not exist
 *            Broker Error 0196 0196: Disconnected due to new location
 * -------------------------------------------------------------------
 *
 * </pre></blockquote><hr>
 *
 * @author  Tony Blanco
 * @version 1.0, 01/02/2002
 */
public final class EntireXBroker
{
//	public static final String ERROR_0003_CODE    = "0004";
//	public static final String ERROR_0003_MESSAGE = "Error in return response.";


	private static Logger logger = LogManager.getLogger( EntireXBroker.class.getName() );  // PBSC palermor IR84565

	// this is a static number used to maintain a unique identifier
	// for each instance of this broker object ... this counter is
	// used with(concatenated with) the connection "userId" name
	//
	private static int counter = 0;

	// broker communication information
	//
	private String  brokerID     = "";
	private String  serverClass  = "";
	private String  serverName   = "";
	private String  service      = "";
	private String  userId       = "";
	private boolean isUOW        = false;
	private String  uowTimeout   = "2M";
	private int     uowMaxLength = 0;
	private String  response     = "";


//	public static void
//	main( String [] args )
//	{
//		///////////////////////////////////////////////////////////////
//		// load all property files
//		//
//		PropertyManager.init( "D:\\code\\projects\\fccsc\\manager\\properties" );
//		PropertyManager.loadAll();
//
//		///////////////////////////////////////////////////////////////
//		// load the log4j properties
//		//
//		Properties propLog4j = (Properties) PropertyManager.getProperty( "log4j.properties" );
//		PropertyConfigurator.configure( propLog4j );
//
//
//		for ( int i = 0; i < 2; i++)
//		{
//			new Thread()
//			{
//				String data =  "" +
//				"000736TCP FEDI00FLATRANFORMFLACENTSERV04200                 " +
//				"        000001REQUEST Y\n" +
//				"        000010PROCESS VERIFY    \n" +
//				"        000016STATEKEYT0000531M0000004\n" +
//				"        000015IPADDRES10.0.0.15      \n" +
//				"        000005PORTNUMB03006\n" +
//				"        000050RETNDATAWA00,                           SWWP0F00P0001267  \n" +
//				"        000030HTMLKEY                               \n" +
//		//		//"        000012STUID   123456789   \n" +
//				"        000012STUID   265155136   \n" +
//		//		//"        000015PIN/PW  1111           \n" +
//				"        000015PIN/PW  1254           \n" +
//				"        000001WAIT    N\n" +
//				"        000005RETNCODE00000\n" +
//				"        000240RETNMESS                                                                                                                                                                                                                                                \n" +
//				"";
//
//			    public void run()
//		        {
//					try
//					{
//						//System.out.println( "running [" + this.getName() + "] START" );
//						Properties prop = (Properties) PropertyManager.getProperty( "message.VERIFY.properties" );
//
//						EntireXBroker broker = new EntireXBroker( prop );
//						broker.sendMessage( data );
//
//						// get response message
//						String response = broker.getResponse();
//
//						System.out.println( "response [" + this.getName() + "] size [" + response.length() + "]" );
//						//System.out.println( "response      [" + response + "]" );
//						//System.out.println( "running [" + this.getName() + "] END" );
//					}
//					catch (BrokerException ex)
//					{
//						ex.printStackTrace();
//					}
//		        }
//			}.start();
//	    }
//
//		//ControlBlock      ediCB = new ControlBlock(      response );
//		//StandardDataBlock ediSB = new StandardDataBlock( response );
//		//System.out.println( "CB [" + ediCB + "]" );
//		//System.out.println( "SB [" + ediSB + "]" );
//	}


	/**
	 * This class encapsulates the ability to communicate with the
	 * Entire X Broker on the mainframe using the Entire X Java
	 * interface client.
	 *
	 * Parameters to instantiate this object are loaded from a properties file.
	 *
	 * @param p_prop a properties object.
	 */
	public
	EntireXBroker( Properties p_prop )
	{
		this(   p_prop.getProperty( "entirex.brokerid"      ),
				p_prop.getProperty( "entirex.serverclass"   ),
				p_prop.getProperty( "entirex.servername"    ),
				p_prop.getProperty( "entirex.service"       ),
				p_prop.getProperty( "entirex.userid"        ),
				 new Boolean(
						p_prop.getProperty( "entirex.uow" ) ).booleanValue(),
						p_prop.getProperty( "entirex.uow.timeout"   ),
						Integer.parseInt( p_prop.getProperty( "entirex.uow.maxlength" ) ) );
	}


	/**
	 * This class encapsulates the ability to communicate with the
	 * Entire X Broker on the mainframe using the Entire X Java
	 * interface client.
	 *
	 * @param  p_brokerID     Broker Id ........ etb227:9999
	 * @param  p_serverClass  Server Class ..... WEBAPP
	 * @param  p_serverName   Server Name ...... UOW
	 * @param  p_service      Service .......... SERVER01
	 * @param  p_userId       User Id .......... blah
	 * @param  p_uow          UOW Request ...... true/false
	 * @param  p_uowTimeout   UOW Time Out ..... 90S (seconds) ... 2M (minutes)
	 * @param  p_uowMaxLength UOW Max Length ... 10000
	 */
	public
	EntireXBroker(  String p_brokerID, String p_serverClass, String p_serverName,
					String p_service, String p_userId,
					boolean p_uow, String p_uowTimeout, int p_uowMaxLength )
	{
		this.brokerID     = p_brokerID;
		this.serverClass  = p_serverClass;
		this.serverName   = p_serverName;
		this.service      = p_service;
		this.userId       = p_userId;
		this.isUOW        = p_uow;
		this.uowTimeout   = p_uowTimeout;
		this.uowMaxLength = p_uowMaxLength;

		// check timeout value - default is 2 minutes
		if ( uowTimeout.length() < 0 ) { uowTimeout = "2M"; }

		// bounce the counter
		//synchronized ( this ) { counter++; }
		counter++;
	}


	/**
	 * Returns the Broker Id.
	 * @return the broker Id.
	 */
	public String getBrokerId() { return this.brokerID; }

	/**
	 * Returns the Server Class.
	 * @return the server class.
	 */
	public String getServerClass() { return this.serverClass; }

	/**
	 * Returns the Server Name.
	 * @return the server name.
	 */
	public String getServerName() { return this.serverName; }

	/**
	 * Returns the Service.
	 * @return the service.
	 */
	public String getService() { return this.service; }

	/**
	 * Returns the User Id.
	 * @return the user id.
	 */
	public String getUserId() { return this.userId; }

	/**
	 * Returns if this broker instance is for a Unit Of Work. (default is false)
	 * @return true, if unit of work request; false, if non-unit of work request.
	 */
	public boolean isUOW() { return this.isUOW; }

	/**
	 * Returns the UOW Timeout. (default is 0)
	 * @return the uow timeout.
	 */
	public String getUOWTimeout() { return this.uowTimeout; }

	/**
	 * Returns the UOW Max Recieve Length. (default is 2M)
	 * @return the uow max receive length.
	 */
	public int getUOWMaxLength() { return this.uowMaxLength; }

	/**
	 * Returns the response message received from our transmission.
	 * @return the response message.
	 */
	public String getResponse() { return this.response; }


	/**
	 * Sends a message to broker and waits for a return message response.
	 *
	 * @param p_message data to be sent to broker
	 * @throws BrokerException
	 */
	public void
	sendMessage( String p_message ) throws BrokerException
	{
		StringBuffer  messageReturn = new StringBuffer();
		StringBuffer  messageXml    = new StringBuffer( this.uowMaxLength );
		String        message       = "";
		Vector<String> vMessages    = new Vector<String>();

		UnitofWork    uow           = null;
		BrokerMessage brokerMessage = new BrokerMessage();
		Broker        broker        = null;
		BrokerService brokerService = null;

		try
		{
			//broker = new Broker( brokerID, userId );
			broker = new Broker( brokerID, userId + counter );
			broker.logon();

			brokerService = new BrokerService( broker, serverClass, serverName, service );
			brokerService.setMaxReceiveLen( this.uowMaxLength );

			// ---------------------------------------------------------------
			if ( this.isUOW )
			{
				uow = new UnitofWork( brokerService );
				uow.setLifetime( uowTimeout );

				// ---------------------------------------------------------------
				// break up the large message into smaller pieces
				while ( true )
				{
					if ( p_message.length() < this.uowMaxLength )
					{
						vMessages.addElement( p_message );
						break;
					}
					else
					{
						vMessages.addElement( p_message.substring( 0, this.uowMaxLength - 1 ) );
						p_message = p_message.substring( this.uowMaxLength );
					}
				}

				// ---------------------------------------------------------------
				// send the message pieces one at a time
				// then send a final commit as the last sent message
				for ( int i = 0; i < vMessages.size(); i++ )
				{
					message = vMessages.elementAt( i );
					messageXml.setLength( 0 );
					messageXml.append( message );

					if ( message.length() < this.uowMaxLength )
					{
						for ( int j = 0; j < (this.uowMaxLength - message.length()); j++ )
							{ messageXml.append( ' ' ); }
					}

					brokerMessage.setMessage( messageXml.toString() );
					uow.send( brokerMessage );
				}

				// send complete message transaction
				uow.commit();


				// ------------------------------------------------------------
				// receive the returning messages from Broker
				String messageUow = "";

				while ( true )
				{
					brokerMessage = uow.receive( this.uowTimeout );

					messageReturn.append( brokerMessage.toString() );

					messageUow = uow.getStatus();

					if ( messageUow.equals( "RECV_NONE" ) ||
						 messageUow.equals( "RECV_LAST" ) ||
						 messageUow.equals( "RECV_ONLY" )  )
					{
						uow.commit();
						break;
					}
				}

				this.response = messageReturn.toString();

				// end this conversation
				uow.endConversation();
			}
			else
			{
				// start and send synchroneous conversational requests
				Conversation  conv           = new Conversation( brokerService );
				BrokerMessage brokerResponse = null;

				brokerMessage.setMessage( p_message );
				brokerResponse = conv.sendReceive( brokerMessage, this.uowTimeout );

				this.response = brokerResponse.toString();

				// end this conversation
				conv.end();
		    }
		}
		catch (BrokerException be)
		{
			// [Broker Error 0013 0308: Wrong length for TEXT: User currently active (PCB,=BUSY)        ]
			// [Broker Error 0196 0196: Disconnected due to new location]

			//String format = "Broker Error [CLASS] [CODE] [MESSAGE]";
			String error  =  be.toString();

		    //System.out.println( format );
		    //System.out.println( error );

			//logger.fatal( format );
			logger.fatal( "Communication: {}", error  );  // PBSC palermor IR84565

			throw be;
		}
		finally
		{
			try
			{
				brokerService.endallConversations();  // PBSC palermor IR84565 documentation states that we need to end before logoff/disconnect.
				broker.logoff();
				broker.disconnect();  // PBSC palermor IR84565
			}
			catch ( BrokerException be2 )
			{
				logger.fatal( "Logging off:   {}", be2.toString() );  // PBSC palermor IR84565
				throw be2;
			}
		}
	}
}
