package fccsc.manager.crypto;

/**
 * PBSC History
 * ************
 * PBSC palermor IR84565 05/19/2015 Update to log4j2.
 */

import intarsys.crypto.Crypto;

import java.util.Properties;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;


public class MessageCryptography
{
    private static Logger  logger          = LogManager.getLogger( MessageCryptography.class );  // PBSC palermor IR84565
	private static boolean bInitialized    = false;
	private static String  pathCertificate = "";
	private static String  pathPrivateKey  = "";
	private static String  pathPassPhrase  = "";
	private static String  passPhrase      = "";


	/**
	 * Initializes configuration information.
	 * Property file must contain the following configuration data:
	 *
	 * crypto.path.certificate = cryptography/certificate.crt ...... (relative example)
	 * crypto.path.privatekey  = d:/cryptography/privatekey.pem .... (absolute example)
	 * crypto.path.passphrase  = d:/cryptography/passphrase.pph .... (absolute example)
	 *
	 * Each property path can use a relative or an absolute file path.
	 *
	 * @param p_property a properties object.
	 */
	public static void
	init( Properties p_property )
	{
		init(   p_property.getProperty( "crypto.path.certificate" ),
			    p_property.getProperty( "crypto.path.privatekey"  ),
				p_property.getProperty( "crypto.path.passphrase"  ) );
	}


	/**
	 * Initializes configuration information.
	 *
	 * @param p_pathCertificate a file path to the certifcate.
	 * @param p_pathPrivateKey  a file path to the private key.
	 * @param p_pathPassPhrase  a file path to the encrypted pass phrase.
	 */
	public static void
	init( String p_pathCertificate, String p_pathPrivateKey, String p_pathPassPhrase )
	{
		if ( bInitialized == false )
		{
			pathCertificate = p_pathCertificate;
			pathPrivateKey  = p_pathPrivateKey;
			pathPassPhrase  = p_pathPassPhrase;
			bInitialized    = true;

			decryptPassPhraseFile();

			//System.out.println( "pathCertificate [" + pathCertificate + "]" );
			//System.out.println( "pathPrivateKey  [" + pathPrivateKey  + "]" );
			//System.out.println( "pathPassPhrase  [" + pathPassPhrase  + "]" );
			//System.out.println( "passPhrase      [" + passPhrase      + "]" );
		}
	}



	/**
	 * Encrypts a message using a certificate.
	 * From this (CB + DB) -> (CB + ESK + EDB + SK).
	 *
	 * @param p_data            the data to encrypt.
	 * @param p_pathCertificate the file path to the certificate.
	 * @return  a CryptoData object.
	 */
	public synchronized static CryptoData
	encrypt( String p_data )
	{
		if ( logger.isDebugEnabled() )
		{
			logger.debug( "Encrypting message with certificate ..." );
		}

		byte [] arrData = Crypto.encKeyAndFMsg( p_data, pathCertificate );

		return buildCryptoData( arrData );
	}


	/**
	 * Encrypts a message using a session key.
	 * From this (CB + DB) -> (CB + ESK + EDB).
	 *
	 * @param p_data       the data to encrypt.
	 * @param p_sessionKey the session key.
	 * @return             a byte [] contianing the encrypted data (CB + ESK + EDB).
	 */
	public synchronized static byte []
	encryptWithSessionKey( byte [] p_data, byte [] p_sessionKey )
	{
		if ( logger.isDebugEnabled() )
		{
		    logger.debug( "Encrypting message with session key ..." );
		}

		return Crypto.encryptFMsg( p_data, p_sessionKey );
	}


	/**
	 * Decrypts a message using the private key and pass phrase.
	 * From this (CB + ESK + EDB) -> (CB + DB + SK).
	 *
	 * @param p_data the data to decrypt.
	 * @return       a CryptoData object.
	 */
	public synchronized static CryptoData
	decrypt( byte [] p_data )
	{
		if ( logger.isDebugEnabled() )
		{
		    logger.debug( "Decrypting message with private key and passphrase ..." );
		}

		byte [] arrData = Crypto.decKeyAndFMsg( p_data, pathPrivateKey, passPhrase );

		return buildCryptoData( arrData );
	}


	/**
	 * Decrypts a message using a session key.
	 * From this (CB + ESK + EDB) -> (CB + DB).
	 *
	 * @param p_data the data to decrypt.
	 * @return       a byte [] containing the unencrypted data (CB + DB).
	 */
	public synchronized static byte []
	decryptWithSessionKey( byte [] p_data, byte [] p_sessionKey )
	{
		if ( logger.isDebugEnabled() )
		{
			logger.debug( "Decrypting message with session key ..." );
		}

		return Crypto.decryptFMsg( p_data, p_sessionKey );
	}


	private synchronized static CryptoData
	buildCryptoData( byte [] p_arrData )
	{
		byte [] arrSize       = new byte[ 6 ];
		byte [] arrSessionKey = new byte[ 16 ];

		for ( int i = 0; i < 6; i++ ) { arrSize[ i ] = p_arrData[ i ]; }

		int     iSize   = Integer.parseInt( new String( arrSize ) );
		byte [] arrData = new byte[ iSize ];

		for ( int i = 0; i < iSize; i++ ) { arrData[ i ] = p_arrData[ i ]; }
		for ( int i = 0; i < 16;    i++ )
		{
		    arrSessionKey[ i ] = p_arrData[ iSize + i ];
		}

		return new CryptoData( arrData, arrSessionKey );
	}


	private synchronized static void
	decryptPassPhraseFile()
	{
		if ( logger.isDebugEnabled() )
		{
			logger.debug( "Decrypting pass phrase file ..." );
		}

		PassPhrase phrase = new PassPhrase( pathPassPhrase );
		phrase.fileLoad();

		passPhrase = new String( phrase.getPassPhrase() );
	}
}